using System;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using System.Xml;
using System.Xml.Serialization;

//*****Open issue remains with bool values always being set to true even after cam initialized.
// Timer doesn't seem to work to show images continually, will need to implement the Mightex Hooker callback
// if we want direct updates to the windows form.

namespace LEDDriver
{
    unsafe public struct LEDChannelData
    {
        public int Normal_CurrentMax;
        public int Normal_CurrentSet;
        public int Strobe_CurrentMax;
        public int Strobe_RepeatCnt;
        public fixed int Strobe_Profile[256];
        public int Trigger_CurrentMax;
        public int Trigger_Polarity;
        public fixed int Trigger_Profile[256];
    }

    public class CLEDDriver
    {
        //Members...
        public enum WORK_MODE {DISABLE = 0, NORMAL, STROBE, TRIGGER};

        public LEDChannelData LedChannelData;

        private uint _devHandle = 99;

        private string _ledError = "LED Driver Error";
        
        //private LEDChannelData _ledChannelData = new LEDChannelData();

        //default constructor for testing
        public CLEDDriver(Form1 mightexCamera)
        {
            LedChannelData = new LEDChannelData();
            LedChannelData.Normal_CurrentMax = 20;
            LedChannelData.Normal_CurrentSet = 10;
        }

         /// <summary>
        /// Call this function first, this function communicates with device driver to reserve resources
        /// </summary>
        /// <returns>number of cameras on USB 2.0 chan</returns>
        public int InitDevice()
        {
            int numLEDDriver = MTUSBLEDDriverInitDevices();

            if (numLEDDriver < 0)
            {
                MessageBox.Show("Error trying to initialize sources. No drivers found on USB 2.0 bus.", _ledError, MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            return numLEDDriver;
        }
        
        public void OpenDevice(int deviceID)
        {
            _devHandle = MTUSBLEDDriverOpenDevice(deviceID);
        }

        public void CloseDevice()
        {
            MTUSBLEDDriverCloseDevice(_devHandle);
        }

        public string GetSerialNo()
        {
            string moduleNumber = "Unknown";
            StringBuilder rtnStr = new StringBuilder(32);
            
            if (MTUSBLEDDriverSerialNumber(_devHandle, rtnStr, 32) < 0)
            {
                MessageBox.Show("Error trying to retrieve module number.", _ledError, MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            else
            {
                moduleNumber = rtnStr.ToString();
            }
            return moduleNumber;
        }

        public int GetChannelNumber()
        {
            int chl_no;

            chl_no = MTUSBLEDDriverDeviceChannels(_devHandle);
            return chl_no;
        }


        public int SetChannelMode( int channel, WORK_MODE workMode)
        {
 
            return (MTUSBLEDDriverSetMode( _devHandle, (uint)channel, (uint)workMode ) );
        }


        unsafe public void SetNormalPara( int channel )
        {
            fixed (LEDChannelData * structPtr = &LedChannelData)
            {
                //if (MtxGetFrameSetting(_devHandle, ref _imgControl) < 0)
                if (MTUSBLEDDriverSetNormalPara(_devHandle, (uint)channel, (uint)structPtr) < 0)
                {
                    MessageBox.Show("Error trying to set LED Driver Normal Settings", _ledError, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        unsafe public void SetStrobeProfileData(int step, int value)
        {
            fixed (LEDChannelData* structPtr = &LedChannelData)
            {
                structPtr->Strobe_Profile[step] = value;
            }
        }

        unsafe public void SetStrobePara(int channel)
        {
            fixed (LEDChannelData* structPtr = &LedChannelData)
            {
                //if (MtxGetFrameSetting(_devHandle, ref _imgControl) < 0)
                if (MTUSBLEDDriverSetStrobePara(_devHandle, (uint)channel, (uint)structPtr) < 0)
                {
                    MessageBox.Show("Error trying to set LED Driver Strobe Settings", _ledError, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        unsafe public void SetTriggerProfileData(int step, int value)
        {
            fixed (LEDChannelData* structPtr = &LedChannelData)
            {
                structPtr->Trigger_Profile[step] = value;
            }
        }

        unsafe public void SetTriggerPara(int channel)
        {
            fixed (LEDChannelData* structPtr = &LedChannelData)
            {
                //if (MtxGetFrameSetting(_devHandle, ref _imgControl) < 0)
                if (MTUSBLEDDriverSetTriggerPara(_devHandle, (uint)channel, (uint)structPtr) < 0)
                {
                    MessageBox.Show("Error trying to set LED Driver Trigger Settings", _ledError, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

     
        #region Mightex interop functions for accessing Mightex_LEDDriver_SDK_Stdcall.dll file
  
        //Call this function first, this function communicates with device driver to reserve resources
        //When the system uses NTFS use WINNT, for FAT32 use WINDOWS

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverInitDevices", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverInitDevices();

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverOpenDevice", CallingConvention = CallingConvention.StdCall)]
        private static extern uint MTUSBLEDDriverOpenDevice(int deviceID);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverCloseDevice", CallingConvention = CallingConvention.StdCall)]
        private static extern uint MTUSBLEDDriverCloseDevice(uint devHandle);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverSerialNumber", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverSerialNumber(uint devHandle, StringBuilder moduleNo, uint size);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverDeviceChannels", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverDeviceChannels(uint devHandle);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverDeviceModuleType", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverDeviceModuleType(uint devHandle);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverSetMode", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverSetMode(uint devHandle, uint channel, uint mode);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverSetNormalPara", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverSetNormalPara(uint devHandle, uint channel, uint channelData);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverSetNormalCurrent", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverSetNormalCurrent(uint devHandle, uint channel, uint current);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverSetStrobePara", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverSetStrobePara(uint DevHandle, uint channel, uint channelData );

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverSetTriggerPara", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverSetTriggerPara(uint DevHandle, uint channel, uint channelData);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverResetDevice", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverResetDevice(uint DevHandle);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverStorePara", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverStorePara(uint devHandle );

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverRestoreDefault", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverRestoreDefault(uint devHandle);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverGetLoadVoltage", CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverGetLoadVoltage(uint devHandle, uint channel);

        [DllImport("Mightex_LEDDriver_SDK_Stdcall.dll", EntryPoint = "MTUSB_LEDDriverGetCurrentPara",CallingConvention = CallingConvention.StdCall)]
        private static extern int MTUSBLEDDriverGetCurrentPara(uint DevHandle, uint channel, uint channelData);


        #endregion
    }

}
